<?php
/*+**********************************************************************************
 * LabsMobile Implementation on vtiger 7 plugin (FIXED)
 *
 * - Uses HTTPS
 * - Uses GET for send/ack endpoints
 * - Normalizes Colombia numbers: 10 digits -> prepend 57
 * - PHP 7+: replaces split() with explode()
 * - Safer parsing for query response
 ************************************************************************************/

class SMSNotifier_LabsMobile_Provider implements SMSNotifier_ISMSProvider_Model {

	private $userName;
	private $password;
	private $parameters = array();

	const SERVICE_URI = 'https://api.labsmobile.com';

	private static $REQUIRED_PARAMETERS = array(
		array('name' => 'Username', 'label' => 'Username', 'type' => 'text'),
		array('name' => 'Token', 'label' => 'Token', 'type' => 'password'),
		array('name' => 'Sender', 'label' => 'Sender', 'type' => 'text'),
		array('name' => 'Charset', 'label' => 'Charset GSM or Unicode', 'type' => 'picklist', 'picklistvalues' => array('1' => 'GSM', '2' => 'Unicode'))
	);

	public function getName() {
		return 'LabsMobile';
	}

	public function getRequiredParams() {
		return self::$REQUIRED_PARAMETERS;
	}

	public function getServiceURL($type = false) {
		if($type) {
			switch(strtoupper($type)) {
				case self::SERVICE_AUTH:  return self::SERVICE_URI . '/get/auth.php';
				case self::SERVICE_SEND:  return self::SERVICE_URI . '/get/send.php?';
				case self::SERVICE_QUERY: return self::SERVICE_URI . '/get/ack.php?';
			}
		}
		return false;
	}

	public function setAuthParameters($userName, $password) {
		$this->userName = $userName;
		$this->password = $password;
	}

	public function setParameter($key, $value) {
		$this->parameters[$key] = $value;
	}

	public function getParameter($key, $defaultValue = false) {
		if(isset($this->parameters[$key])) {
			return $this->parameters[$key];
		}
		return $defaultValue;
	}

	/**
	 * SEND
	 */
	public function send($message, $toNumbers) {
		if(!is_array($toNumbers)) {
			$toNumbers = array($toNumbers);
		}

		$toNumbers = $this->cleanNumbers($toNumbers);

		// Si después de limpiar quedan vacíos, cortar y reportar error claro
		if (empty($toNumbers)) {
			return array(array(
				'error' => true,
				'statusmessage' => 'No valid destination numbers after normalization.',
				'status' => self::MSG_STATUS_FAILED
			));
		}

		$clientMessageReference = $this->generateClientMessageReference();
		$response = $this->sendMessage($clientMessageReference, $message, $toNumbers);

		return $this->processSendMessageResult($response, $clientMessageReference, $toNumbers);
	}

	private function generateClientMessageReference() {
		return uniqid();
	}

	/**
	 * NORMALIZACIÓN DE NÚMEROS
	 * - elimina todo lo que no sea dígito
	 * - quita ceros iniciales
	 * - si queda en 10 dígitos (Colombia), antepone 57
	 */
	private function cleanNumbers($numbers) {
		$clean = array();

		foreach ($numbers as $number) {
			$n = preg_replace('/[^\d]/', '', (string)$number);
			$n = ltrim($n, '0');

			// Colombia: 10 dígitos -> 57 + número
			if (strlen($n) === 10) {
				$n = '57' . $n;
			}

			// Acepta solo longitudes razonables (puedes ajustar)
			if (strlen($n) >= 10 && strlen($n) <= 15) {
				$clean[] = $n;
			}
		}

		// Eliminar duplicados por si acaso
		$clean = array_values(array_unique($clean));

		return $clean;
	}

	private function sendMessage($clientMessageReference, $message, $toNumbers) {
		$sender  = $this->getParameter('Sender', '');
		$token   = $this->getParameter('Token', '');
		$username= $this->getParameter('Username', '');
		$charset = $this->getParameter('Charset', '');

		$serviceURL = $this->getServiceURL(self::SERVICE_SEND);

		$serviceURL .= 'username=' . urlencode($username) . '&';
		$serviceURL .= 'password=' . urlencode($token) . '&';
		$serviceURL .= 'msisdn='   . urlencode(implode(',', $toNumbers)) . '&';
		$serviceURL .= 'message='  . urlencode(html_entity_decode($message, ENT_QUOTES, 'UTF-8')) . '&';
		$serviceURL .= 'sender='   . urlencode($sender) . '&';

		if($charset === '2') {
			$serviceURL .= 'ucs2=1&';
		}

		$serviceURL .= 'subid=' . urlencode($clientMessageReference);

		// LOG opcional (descomenta si quieres ver exactamente qué URL se está enviando)
		// file_put_contents('labsmobile_send.log', $serviceURL . PHP_EOL, FILE_APPEND);

		$httpClient = new Vtiger_Net_Client($serviceURL);

		// Endpoint /get/send.php => GET
		$response = $httpClient->doGet(false);

		// LOG opcional de respuesta
		// file_put_contents('labsmobile_send.log', "RESPONSE: " . $response . PHP_EOL, FILE_APPEND);

		return $response;
	}

	private function processSendMessageResult($response, $clientMessageReference, $toNumbers) {
		$results = array();

		foreach ($toNumbers as $number) {
			$result = array();
			$result['to'] = $number;

			// Algunos responses vienen en texto simple con "successfully"
			if (is_string($response) && stripos($response, "successfully") !== false) {
				$result['id'] = $clientMessageReference . '--' . $number;
				$result['status'] = self::MSG_STATUS_PROCESSING;
			} else {
				// Si no dice successfully, devolvemos el response tal cual para ver el error
				$result['error'] = true;
				$result['status'] = self::MSG_STATUS_FAILED;
				$result['statusmessage'] = is_string($response) ? $response : 'Unknown response from LabsMobile';
			}

			$results[] = $result;
		}

		return $results;
	}

	/**
	 * QUERY
	 */
	public function query($messageid) {
		$messageidSplit = explode('--', $messageid);
		$clientMessageReference = isset($messageidSplit[0]) ? trim($messageidSplit[0]) : '';
		$number = isset($messageidSplit[1]) ? trim($messageidSplit[1]) : '';

		$response = $this->queryMessage($clientMessageReference, $number);

		// La firma real solo recibe $response
		return $this->processQueryMessageResult($response);
	}

	private function queryMessage($clientMessageReference, $number) {
		$token = $this->getParameter('Token', '');
		$username = $this->getParameter('Username', '');

		$serviceURL = $this->getServiceURL(self::SERVICE_QUERY);
		$serviceURL .= 'username=' . urlencode($username) . '&';
		$serviceURL .= 'password=' . urlencode($token) . '&';
		$serviceURL .= 'subid='    . urlencode($clientMessageReference) . '&';
		$serviceURL .= 'msisdn='   . urlencode($number);

		// LOG opcional
		// file_put_contents('labsmobile_query.log', $serviceURL . PHP_EOL, FILE_APPEND);

		$httpClient = new Vtiger_Net_Client($serviceURL);

		// Endpoint /get/ack.php => GET
		$response = $httpClient->doGet(false);

		// LOG opcional
		// file_put_contents('labsmobile_query.log', "RESPONSE: " . $response . PHP_EOL, FILE_APPEND);

		return $response;
	}

	private function processQueryMessageResult($response) {
		$result = array();
		$result['needlookup'] = 1;

		if (!is_string($response) || trim($response) === '') {
			$result['error'] = true;
			$result['statusmessage'] = 'Empty response from LabsMobile ack endpoint.';
			$result['status'] = self::MSG_STATUS_FAILED;
			return $result;
		}

		// LabsMobile suele responder XML; si no es XML, devolvemos el texto
		try {
			$xmlNode = new SimpleXMLElement($response, LIBXML_NOCDATA);
		} catch (Exception $e) {
			$result['error'] = true;
			$result['statusmessage'] = $response; // devolver texto crudo para depurar
			$result['status'] = self::MSG_STATUS_FAILED;
			return $result;
		}

		$status = (string)$xmlNode->status;
		$desc   = isset($xmlNode->desc) ? (string)$xmlNode->desc : '';

		switch($status) {
			case 'error':
				$result['error'] = true;
				$result['statusmessage'] = $desc !== '' ? $desc : 'LabsMobile status=error';
				$result['status'] = self::MSG_STATUS_FAILED;
				break;

			case 'processed':
			case 'operator':
			case 'gateway':
				$result['error'] = false;
				$result['status'] = self::MSG_STATUS_PROCESSING;
				break;

			case 'handset':
				$result['error'] = false;
				$result['status'] = self::MSG_STATUS_DELIVERED;
				break;

			default:
				$result['error'] = false;
				$result['status'] = self::MSG_STATUS_PROCESSING;
				break;
		}

		return $result;
	}

	function getProviderEditFieldTemplateName() {
		return 'BaseProviderEditFields.tpl';
	}
}
?>

